# -*- coding: utf-8 -*-
"""
@Project: LinalgDat2022
@File: AdvancedExtensions.py

@Description: Project C Determinant and Gram-Schmidt extensions.

"""

import math
import sys

sys.path.append('../Core')
from Vector import Vector
from Matrix import Matrix

Tolerance = 1e-6


def SquareSubMatrix(A: Matrix, i: int, j: int) -> Matrix:
    """
    This function creates the square submatrix given a square matrix as
    well as row and column indices to remove from it.

    Remarks:
        See page 246-247 in "Linear Algebra for Engineers and Scientists"
        by K. Hardy.

    Parameters:
        A:  N-by-N matrix
        i: int. The index of the row to remove.
        j: int. The index of the column to remove.

    Return:
        The resulting (N - 1)-by-(N - 1) submatrix.
    """
    M = A.M_Rows
    B = A.asArray()
    B.pop(i)
    for x in range(0,M-1):
        B[x].pop(j)
    return Matrix.fromArray(B)




def Determinant(A: Matrix) -> float:
    """
    This function computes the determinant of a given square matrix.

    Remarks:
        * See page 247 in "Linear Algebra for Engineers and Scientists"
        by K. Hardy.
        * Hint: Use SquareSubMatrix.

    Parameter:
        A: N-by-N matrix.

    Return:
        The determinant of the matrix.
    """
    det = 0
    N = A.N_Cols
    if N == 1:
        return A[0,0]
    for j in range(0,N):    
        det += A[0,j] * (-1)**j * Determinant(SquareSubMatrix(A, 0, j))
    return det


def VectorNorm(v: Vector) -> float:
    """
    This function computes the Euclidean norm of a Vector. This has been implemented
    in Project A and is provided here for convenience

    Parameter:
         v: Vector

    Return:
         Euclidean norm, i.e. (\sum v[i]^2)^0.5
    """
    nv = 0.0
    for i in range(len(v)):
        nv += v[i]**2
    return math.sqrt(nv)


def SetColumn(A: Matrix, v: Vector, j: int) -> Matrix:
    """
    This function copies Vector 'v' as a column of Matrix 'A'
    at column position j.

    Parameters:
        A: M-by-N Matrix.
        v: size M vector
        j: int. Column number.

    Return:
        Matrix A  after modification.

    Raise:
        ValueError if j is out of range or if len(v) != A.M_Rows.
    """
    M = A.M_Rows
    B = A
    if len(v) != M:
        raise ValueError()
    for x in range(M):
        B[x,j] = v[x]
    return B

def Transpose(A: Matrix) -> Matrix:
    """
    Computes the transpose of a given Matrix.

    See page 69 in "Linear Algebra for Engineers and Scientists"
    by K. Hardy.

    :param A: A M-by-N Matrix.
    :returns: A N-by-M Matrix B such that B = A^T.
    """
    M = A.M_Rows
    N = A.N_Cols
    B = Matrix(N, M)
    for i in range(N):
        for j in range(M):
            B[i,j] = A[j,i]
    return B

def VecProduct(v1: Vector, v2: Vector):
    prod = 0
    for j in range(v1.size()):
        prod += v1[j]*v2[j]
    return prod

def Zeroq(q):
    for j in range(q.size()):
        if q[j] != 0:
            return False
    return True


def GramSchmidt(A: Matrix) -> tuple:
    """
    This function computes the Gram-Schmidt process on a given matrix.

    Remarks:
        See page 229 in "Linear Algebra for Engineers and Scientists"
        by K. Hardy.

    Parameter:
        A: M-by-N matrix. All columns are implicitly assumed linear
        independent.

    Return:
        tuple (Q,R) where Q is a M-by-N orthonormal matrix and R is an
        N-by-N upper triangular matrix.
    """
    N = A.N_Cols
    M = A.M_Rows
    R = Matrix(N,N)
    Q = Matrix(M,N)

    for j in range(0,N):
        u = A.Column(j)
        q = u
        for i in range(0,j):
            R[i,j] = VecProduct(Q.Column(i), u)
            q = q - R[i,j] * Q.Column(i)
        if Zeroq(q):
            break
        R[j,j] = VectorNorm(q)
        for x in range(q.size()):
            q[x] = q[x] * (1/R[j, j])
        Q = SetColumn(Q, q, j)
    return((Q,R))